-- WorldViewIconMan
-- Author: Zur13
-- DateCreated: 1/20/2019 6:19:07 PM
-- Copyright 2019, Zur13, All rights reserved
--------------------------------------------------------------

include( "InstanceManager" );

local MAX_ESTIMATION_RADIUS = 30;

local OWNED_PLOT = -8; -- in yields

local LUX_RES_VAL = 24; -- in yields
local STR_RES_VAL = 16; -- in yields

local FOOD_COEF = 1.10; -- food importance
local PROD_COEF = 0.90; -- production importance
local OTHE_COEF = 0.70; -- other yields importance

local FRESH_WATER_COEF = 0.24; -- importance of fresh water city center
local COASTAL_COEF = 0.19; -- importance of coastal city center

local MAX_RANGE_CONCURRENT_PLOT_COEF = -0.4; -- importance of tile bonuses if tile might be owned by another city and in range of 3 of estimated city center tile
local FAR_RANGE_CONCURRENT_PLOT_COEF = -0.5; -- importance of tile bonuses if tile might be owned by another city and in range of 2 of estimated city center tile
local NEAR_RANGE_CONCURRENT_PLOT_COEF = -0.5 -- importance of tile bonuses if tile might be owned by another city and in range of 1 of estimated city center tile

local KEY_PINDEX	:string = "plotIndex";

local instMan		:table = InstanceManager:new( "RmtIconSetInstance",	"RmtAnchor", Controls.RmtIconContainer );
local mapIcons		:table = {};

-- Stores a list of plot indexes currently showing settlement recommendations
-- Used to efficiently clear settlement recommendations
local usedPlotIds	:table = {};


local texCityOffsetX, texCityOffsetY, texCitySheet = IconManager:FindIconAtlas("ICON_UNITOPERATION_FOUND_CITY", 38);

IsChangedMapParent = false;

--************************************************************
-- Place pin container on the map Z level
local function ChangeParentMap()
	if not IsChangedMapParent then
		local wvc:table = ContextPtr:LookUpControl("/InGame/WorldViewControls"); -- world view
		if wvc ~= nil then
			Controls.RmtIconContainer:ChangeParent(wvc);
			--tPanRightStack:AddChildAtIndex(Controls.RmtLaunchBarBtn, 3);
			--tPanRightStack:CalculateSize();
			--tPanRightStack:ReprocessAnchoring();
			IsChangedMapParent = true;
		end
	end
end

--************************************************************
-- Callaback of the load game UI event
local function OnLoadGameViewStateDone()
	ChangeParentMap();
end

--******************************************************************************
local function GetInstanceAt(plotIndex)
	local pInstance = mapIcons[plotIndex];
	if (pInstance == nil) then
		pInstance = instMan:GetInstance();
		mapIcons[plotIndex] = pInstance;

		local worldX, worldY = UI.GridToWorld( plotIndex );
		pInstance.RmtAnchor:SetWorldPositionVal( worldX, worldY-10.0, 0.0 );
				
		pInstance[KEY_PINDEX] = plotIndex;
	end
	return pInstance;
end

--******************************************************************************
-- Handle the UI shutting down.
local function OnShutdown()
	mapIcons = {};
	instMan:DestroyInstances();
end

--******************************************************************************
local function GetRecommendedLocations(targetPlotId, targetRadius, ignoreVisibility)
	local res = {};
	if targetPlotId == nil then
		return res;
	end
	
	local localPlayer   :number = Game.GetLocalPlayer();
	local localPlayerVis:table = PlayersVisibility[localPlayer];

	local plot = Map.GetPlotByIndex(targetPlotId);
	if plot ~= nil then
		-- const
		local radius = math.min( MAX_ESTIMATION_RADIUS, targetRadius); -- eval radius
		local cityRadius = 4; -- city radius
		local cRadius = radius + cityRadius + cityRadius - 2; -- radius with cities which may affect
		--print("GetRecommendedLocations 1111 ", radius, cityRadius, cRadius);
		--local localPlayer   :number = Game.GetLocalPlayer();

		local selPlot = plot;
		local selPlotX:number = selPlot:GetX();
		local selPlotY:number = selPlot:GetY();

		local radiusPlots = {}; -- all plot instances in radius
		
		local radiusPlotsNO = {}; -- all plot instances in/near radius not owned by another cities
		local radiusPlotsOW = {}; -- all plot instances in/near radius owned by another cities
		local radiusPlotsAF = {}; -- all plot instances in/near radius which may be owned by another cities but not owned yet
		
		local radiusPlotsSRS = {}; -- strategic resources on plot (key == plot id, val == resource)
		local radiusPlotsLRS = {}; -- luxury resource on plot (key == plot id, val == resource)

		local radiusPlotsFD = {}; -- plot yield food output
		local radiusPlotsPD = {}; -- plot yield production output
		local radiusPlotsOT = {}; -- plot yields other (gold, culture, relig, science)

		local radiusPlotsFW = {}; -- plot has fresh water
		local radiusPlotsCW = {}; -- plot is coastal water
		
		local radiusCityPlots = {}; -- city center plots which may affect plots in radius

		-- find all city plots which may affect radius
		for x = selPlotX - cRadius, selPlotX + cRadius, 1 do
		for y = selPlotY - cRadius, selPlotY + cRadius, 1 do
			local iPlot:table = Map.GetPlot(x, y);
			if iPlot ~= nil then
				local idx = iPlot:GetIndex();
				if iPlot:IsCity() then
					radiusCityPlots[idx] = iPlot;
				end
			end
		end
		end
		--print( "    radiusCityPlots  ", table.count(radiusCityPlots) );


		for x = selPlotX - cRadius, selPlotX + cRadius, 1 do
		for y = selPlotY - cRadius, selPlotY + cRadius, 1 do
			local iPlot:table = Map.GetPlot(x, y);
			if iPlot ~= nil then
				local idx = iPlot:GetIndex();
				local iX:number = iPlot:GetX();
				local iY:number = iPlot:GetY();
			
				local dist2:number = Map.GetPlotDistance(iX, iY, selPlotX, selPlotY);

				if dist2 ~= nil and dist2 >= 0 and dist2 <= radius then
					radiusPlots[idx] = iPlot;
				end
					
				-- use plots outside of radius to evaluate city locations in radius
				if iPlot:IsCity() or iPlot:IsOwned() then
					radiusPlotsOW[idx] = iPlot;
				else
					radiusPlotsNO[idx] = iPlot;
				end

				if iPlot:IsFreshWater() then
					radiusPlotsFW[idx] = iPlot;
				elseif iPlot:IsCoastalLand() then
					radiusPlotsCW[idx] = iPlot;
				end

				local resType = iPlot:GetResourceType();
				if resType ~= nil then
					local res = GameInfo.Resources[resType];
					if res ~= nil and res.ResourceClassType == "RESOURCECLASS_LUXURY" then
						radiusPlotsLRS[idx] = res;
					end	

					if res ~= nil and res.ResourceClassType == "RESOURCECLASS_STRATEGIC"  then
						radiusPlotsSRS[idx] = res;
					end	
				end

				local isInCityRange = false; -- iPlot in another city range
				-- check if iPlot is in another city range
				for j, cPlot in pairs(radiusCityPlots) do 
					local cX:number = cPlot:GetX();
					local cY:number = cPlot:GetY();

					local dist:number = Map.GetPlotDistance(iX, iY, cX, cY);
					if dist ~= nil and dist >= 0 and dist < cityRadius then
						isInCityRange = true;
						radiusPlotsAF[idx] = iPlot;
						break;
					end
				end

				-- yields
				local yieldF = iPlot:GetYield( YieldTypes.FOOD );
				local yieldP = iPlot:GetYield( YieldTypes.PRODUCTION );
				local yieldG = iPlot:GetYield( YieldTypes.GOLD );
				local yieldS = iPlot:GetYield( YieldTypes.SCIENCE );
				local yieldC = iPlot:GetYield( YieldTypes.CULTURE );
				local yieldFA = iPlot:GetYield( YieldTypes.FAITH );

				radiusPlotsFD[idx] = yieldF; -- plot yield food output
				radiusPlotsPD[idx] = yieldP; -- plot yield production output
				radiusPlotsOT[idx] = yieldG + yieldS + yieldC + yieldFA; -- plot yields other (gold, culture, relig, science)
			end
		end
		end

		--print( "    radiusPlots  ", table.count(radiusPlots) );
		--print( "    radiusPlotsNO  ", table.count(radiusPlotsNO) );
		--print( "    radiusPlotsAF  ", table.count(radiusPlotsAF) );
		--print( "    radiusPlotsLRS  ", table.count(radiusPlotsLRS) );

		--for idxx,iPlot in pairs(radiusCityPlots) do 
			---- ############################################################################################################################################ DEBUG
			----local idx = iPlot:GetIndex();
			--local pRecommendedPlotInstance = GetInstanceAt(idxx);
			--pRecommendedPlotInstance.FlagBase:SetToolTipString("City plot");
			--pRecommendedPlotInstance.RmtPrevBackground:SetHide(false);
--
			--table.insert(usedPlotIds, idxx);
--
			----print( "              |_radiusCityPlots  ", idxx );
		--end
--
		--
		--for idxx,iPlot in pairs(radiusPlotsAF) do 
			---- ############################################################################################################################################ DEBUG
			----local idx = iPlot:GetIndex();
			--local pRecommendedPlotInstance = GetInstanceAt(idxx);
			--pRecommendedPlotInstance.FlagBase:SetToolTipString("Affected plot");
			--pRecommendedPlotInstance.RmtPrevBackground:SetHide(false);
			--table.insert(usedPlotIds, idxx);
		--end

		-- evaluate

		for i,iPlot in pairs(radiusPlots) do 
			-- for each plot in radius
			local idx = iPlot:GetIndex();
			local iX:number = iPlot:GetX();
			local iY:number = iPlot:GetY();

			local eVal = 0; -- city center plot estimated value
			local availableLux = {}; -- luxury resources available for this city center (key - res type; val - res count)
			local availableStrat = {}; -- strategic resources available for this city center

			local availableFood = 0; -- available food yields (real, due to coefficient)
			local availableProd = 0; -- available production yields (real, due to coefficient)
			local availableOthe = 0; -- available other yields (real, due to coefficient)
			
			local visible = true;
			if not ignoreVisibility and localPlayerVis ~= nil then
				visible = localPlayerVis:IsRevealed(iX, iY);
			end

			---- ############################################################################################################################################ DEBUG
			--print("       evaluate plot", 1, idx)
			--local pRecommendedPlotInstance = GetInstanceAt(idx);
			--pRecommendedPlotInstance.FlagBase:SetToolTipString( 
				--"Eval plot, not isWater: " .. ( not iPlot:IsWater() and 'true' or 'false')  
				--.. "; not isImpassable: " .. ( not iPlot:IsImpassable() and 'true' or 'false') 
				--.. "; radiusPlotsNO: " .. (radiusPlotsNO[idx] ~= nil and 'true' or 'false') 
				--.. "; radiusPlotsAF: " .. (radiusPlotsAF[idx] == nil and 'true' or 'false') 
				--.. "; radiusPlotsLRS: " .. (radiusPlotsLRS[idx] == nil and 'true' or 'false') 
				--);
			--pRecommendedPlotInstance.RmtPrevBackground:SetHide(false);
			--table.insert(usedPlotIds, idx);
			--print("       evaluate plot", 2, idx)

			if radiusPlotsNO[idx] ~= nil and radiusPlotsAF[idx] == nil and radiusPlotsLRS[idx] == nil and not iPlot:IsImpassable() and not iPlot:IsWater() then 
				-- only estimate as possible new ciy center the not owned tiles and not in range of another city and not the lux resource tile 
				-- and of course tile should be passable and not water tile

				local dist = {};
				dist[0] = {};
				dist[1] = {};
				dist[2] = {};
				dist[3] = {};
				--print( "    Eval plot ----  ", 1, iX, iY );
				for x = iX - cityRadius, iX + cityRadius, 1 do
				for y = iY - cityRadius, iY + cityRadius, 1 do
					local rPlot:table = Map.GetPlot(x, y);
					if rPlot ~= nil then
						local rIdx = rPlot:GetIndex();
						local rX:number = rPlot:GetX();
						local rY:number = rPlot:GetY();
			
						local dist2:number = Map.GetPlotDistance(iX, iY, rX, rY);
						
						if dist2 ~= nil and dist2 >= 0 and dist2 < cityRadius then
							dist[dist2][rIdx] = rPlot;
						end
					end
				end
				end
				--print( "    Eval plot ----  ", 2, table.count(dist[0]), table.count(dist[1]), table.count(dist[2]), table.count(dist[3]) );

				for rDist, rPlots in pairs(dist) do
				for rIdx, rPlot in pairs(rPlots) do
					-- if iPlot will be city center then estimate all plots around in city radius not owned by another cities

					if radiusPlotsOW[rIdx] ~= nil then
						-- owned plot in range
						eVal = eVal + OWNED_PLOT;
					else
						--calculate eVal only on not owned plots
						local coeff = 1.0; 
						if radiusPlotsAF[rIdx] ~= nil then
							if rDist == 3 then
								coeff = coeff + MAX_RANGE_CONCURRENT_PLOT_COEF;  -- coefficient for estimating city range tile values which are in range of another city on in range of 3 of possible city center
							elseif rDist == 2 then
								coeff = coeff + FAR_RANGE_CONCURRENT_PLOT_COEF;  -- coefficient for estimating city range tile values which are in range of another city on in range of 2 of possible city center
							elseif rDist == 1 then
								coeff = coeff + NEAR_RANGE_CONCURRENT_PLOT_COEF; -- coefficient for estimating city range tile values which are in range of another city on in range of 1 of possible city center
							end
						end
						
						if radiusPlotsFW[rIdx] ~= nil then
							coeff = coeff + FRESH_WATER_COEF;
						elseif radiusPlotsCW[rIdx] ~= nil then
							coeff = coeff + COASTAL_COEF;
						end

						--print( "             |-------------  coef", coeff);

						local pLux = radiusPlotsLRS[rIdx];
						if pLux ~= nil then
							if availableLux[pLux.ResourceType] == nil then
								availableLux[pLux.ResourceType] = 1;
									
								eVal = eVal + LUX_RES_VAL * coeff;
							else 
								availableLux[pLux.ResourceType] = availableLux[pLux.ResourceType] + 1;

								if availableLux[pLux.ResourceType] == 2 then
									-- second resource of same type values less
									eVal = eVal + LUX_RES_VAL * coeff * 0.15;
								end
							end
						end
					
						--print( "             |-------------  ", 1);

						local pStra = radiusPlotsSRS[rIdx];
						if pStra ~= nil then
							if availableStrat[pStra.ResourceType] == nil then
								availableStrat[pStra.ResourceType] = 1;
									
								eVal = eVal + STR_RES_VAL * coeff;
							else 
								availableStrat[pStra.ResourceType] = availableStrat[pStra.ResourceType] + 1;

								if availableStrat[pStra.ResourceType] == 2 then
									-- second resource of same type values less
									eVal = eVal + STR_RES_VAL * coeff * 0.65;
								end
							end
						end

						--print( "             |-------------  ", 2);
					
					

						if radiusPlotsFD[rIdx] ~= nil then
							availableFood = availableFood + radiusPlotsFD[rIdx] * coeff * FOOD_COEF;
						end

						if radiusPlotsPD[rIdx] ~= nil then
							availableProd = availableProd + radiusPlotsPD[rIdx] * coeff * PROD_COEF;
						end

						if radiusPlotsOT[rIdx] ~= nil then
							availableOthe = availableOthe + radiusPlotsOT[rIdx] * coeff * OTHE_COEF;
						end
					end

				end
				end -- for rDist, rPlots in pairs(dist) do

				-- finish tile estimation
				if visible and availableFood >= 6 and availableProd >= 4 then
					eVal = eVal + availableFood + availableProd + availableOthe;
					local plotInfo = {};
					plotInfo["PI_EVAL"] = eVal;
					plotInfo["PI_PLOT"] = iPlot;
					plotInfo.SettlingLocation = idx;
					table.insert(res, plotInfo);

					--print( "             |-------------  ", 3, eVal, availableFood, availableProd);
				end

				

			end --if radiusPlotsAF[idx] == nil 
		end

		--print( "             |-------------  ", 4);

		-- sort result by val
		local sortFunc = function(a, b) 
			return a.PI_EVAL > b.PI_EVAL;
		end
		table.sort(res, sortFunc);

		--print( "             |-------------  ", 5);
	end

	return res;
end

--******************************************************************************
local function ClearSettlementRecommendations()
	-- Hide previous recommendations
	for i,plotIndex in ipairs(usedPlotIds) do
		local pRecommendedPlotInstance = GetInstanceAt(plotIndex);
		pRecommendedPlotInstance.RmtImprovementRecommendationBackground:SetHide(true);
		pRecommendedPlotInstance.RmtPrevBackground:SetHide(true);
	end

	-- Clear table
	usedPlotIds = {};

	mapIcons = {};
	instMan:DestroyInstances();
end


--******************************************************************************
function OnMapPinFlagLeftClick( plotId )
	--print("OnMapPinFlagLeftClick", plotId)
	local plot = Map.GetPlotByIndex( plotId );
	--UI.LookAtPlot( plot );
	LuaEvents.RmtMapPinLClick( plotId );
end

--******************************************************************************
function OnMapPinFlagRightClick( plotId )
	--print("OnMapPinFlagRightClick", plotId);
	LuaEvents.RmtMapPinRClick( plotId );
end

--******************************************************************************
local function AddSettlementRecommendations( targetPlotId, targetPlotIds, targetRadius, enableSuggestions, multiRadius, ignoreVisibility )
	--print( " ################ AddSettlementRecommendations", 1, targetPlotId, targetPlotIds );
	for i = #targetPlotIds, 1, -1 do
		local plotId = targetPlotIds[i]
		local isMain = plotId == targetPlotId;
		
		if plotId ~= targetPlotId and ( multiRadius or i == 2 ) then
			local pRecommendedPlotInstance = GetInstanceAt(plotId);

			-- Show recommendation and add to list for clean up later
			local yields = "";
			local resources = "";

			if ExposedMembers.MOD_RMT ~= nil then
				yields = Locale.Lookup("LOC_RMT_RAD_PIN_YIELDS_TT")

				local counter;
				counter = ExposedMembers.MOD_RMT.RmtGetYieldsCountString( YieldTypes.FOOD, plotId, targetPlotId );
				if counter == nil then
					counter = " ? "
				end
				yields = yields .. "[NEWLINE][ICON_FOOD] " .. counter;
			
				counter = ExposedMembers.MOD_RMT.RmtGetYieldsCountString( YieldTypes.PRODUCTION, plotId, targetPlotId );
				if counter == nil then
					counter = " ? "
				end
				yields = yields .. " - [ICON_Production] " .. counter;
						
				counter = ExposedMembers.MOD_RMT.RmtGetYieldsCountString( YieldTypes.GOLD, plotId, targetPlotId );
				if counter == nil then
					counter = " ? "
				end
				yields = yields .. " - [ICON_Gold] " .. counter;

			
				counter = ExposedMembers.MOD_RMT.RmtGetYieldsCountString( YieldTypes.SCIENCE, plotId, targetPlotId );
				if counter == nil then
					counter = " ? "
				end
				yields = yields .. "[NEWLINE][ICON_Science] " .. counter;

			
				counter = ExposedMembers.MOD_RMT.RmtGetYieldsCountString( YieldTypes.CULTURE, plotId, targetPlotId );
				if counter == nil then
					counter = " ? "
				end
				yields = yields .. " - [ICON_Culture] " .. counter;

			
				counter = ExposedMembers.MOD_RMT.RmtGetYieldsCountString( YieldTypes.FAITH, plotId, targetPlotId );
				if counter == nil then
					counter = " ? "
				end
				yields = yields .. " - [ICON_Faith] " .. counter;

				resources = ExposedMembers.MOD_RMT.RmtGetResCountString( plotId, targetPlotId );
			end
			--yields = yields .. " - " ..			LuaEvents.RmtGetYieldsCountString( YieldTypes.PRODUCTION, plotId, targetPlotId );
			--yields = yields .. " - " ..			LuaEvents.RmtGetYieldsCountString( YieldTypes.GOLD, plotId, targetPlotId );
			--yields = yields .. "[NEWLINE] " ..  LuaEvents.RmtGetYieldsCountString( YieldTypes.SCIENCE, plotId, targetPlotId );
			--yields = yields .. " - " ..			LuaEvents.RmtGetYieldsCountString( YieldTypes.CULTURE, plotId, targetPlotId );
			--yields = yields .. " - " ..			LuaEvents.RmtGetYieldsCountString( YieldTypes.FAITH, plotId, targetPlotId );
			

			pRecommendedPlotInstance.RmtFlagBtn:SetToolTipString( Locale.Lookup("LOC_RMT_RAD_PIN_TT", i, yields, resources ) );
			pRecommendedPlotInstance.RmtPrevBackground:SetHide(false);

			--print("OnMapPinFlagRightClick", plotId);

			pRecommendedPlotInstance.RmtFlagBtn:SetVoid1( plotId );
			--pRecommendedPlotInstance.RmtFlagBtn:SetVoid2(  );
			pRecommendedPlotInstance.RmtFlagBtn:RegisterCallback( Mouse.eLClick, OnMapPinFlagLeftClick );
			pRecommendedPlotInstance.RmtFlagBtn:RegisterCallback( Mouse.eRClick, OnMapPinFlagRightClick );

			table.insert(usedPlotIds, plotId);
		end
	end

	if enableSuggestions then
		--local pLocalPlayer:table = Players[Game.GetLocalPlayer()];
		local pSettlementRecommendations = GetRecommendedLocations(targetPlotId, targetRadius, ignoreVisibility);
		
		--print( " ################ AddSettlementRecommendations", 12, table.count( pSettlementRecommendations ) );
		
		local cnt = 4;

		if targetRadius > 7 then
			cnt = 8;
		end

		if targetRadius > 18 then
			cnt = 16;
		end
		cnt = math.min( cnt, math.floor( table.count(pSettlementRecommendations) / 3 ) );
		for key, value in ipairs(pSettlementRecommendations) do
			local setPlot = Map.GetPlotByIndex(value.SettlingLocation);
			local setPlotX:number = setPlot:GetX();
			local setPlotY:number = setPlot:GetY();
				
			--print( " ################ AddSettlementRecommendations", 3, dist );
			local pRecommendedPlotInstance = GetInstanceAt(value.SettlingLocation);

			-- Update icon ICON_CIVILIZATION_UNKNOWN
			--local textureOffsetX, textureOffsetY, textureSheet = IconManager:FindIconAtlas("ICON_UNITOPERATION_FOUND_CITY", 38);						
			pRecommendedPlotInstance.RmtImprovementRecommendationIcon:SetTexture(texCityOffsetX, texCityOffsetY, texCitySheet);

			-- Update tooltip
			--pRecommendedPlotInstance.RmtImprovementRecommendationIcon:SetToolTipString(Locale.Lookup("LOC_TOOLTIP_SETTLEMENT_RECOMMENDATION") .. ". [NEWLINE]Icon added by Radial Measuring Tool mod and it may differ from the game recommendations shown when the Settler unit selected. [NEWLINE]Estimated location value: " .. value.PI_EVAL);				
			pRecommendedPlotInstance.RmtImprovementRecommendationIcon:SetToolTipString( Locale.Lookup("LOC_RMT_REC_PIN_TT") .. value.PI_EVAL );				

			-- Show recommendation and add to list for clean up later
			pRecommendedPlotInstance.RmtImprovementRecommendationBackground:SetHide(false);
			table.insert(usedPlotIds, value.SettlingLocation);
			--print( " ################ AddSettlementRecommendations", 4, value.PI_EVAL );
					
			cnt = cnt - 1;
			if cnt <= 0 then
				break;
			end
		end -- for
	end -- if
end

--******************************************************************************
local function Initialize()	

	print( " ################ Start Initializing Mod RMT UI Map Icons Script... ################ " );
	--ContextPtr:SetInitHandler(OnContextInitialize);
	ContextPtr:SetShutdown( OnShutdown );
		
	Events.LoadGameViewStateDone.Add(OnLoadGameViewStateDone);
	
	LuaEvents.RmtClearSettlementRecommendations.Add( ClearSettlementRecommendations );
	LuaEvents.RmtAddSettlementRecommendations.Add( AddSettlementRecommendations );
	LuaEvents.GetRecommendedLocations.Add( GetRecommendedLocations ); -- for fellow modders to integrate this function into their mods
	
	LuaEvents.RmtWVIMInitialized( );

	--Events.LocalPlayerChanged.Add(OnLocalPlayerChanged);

	print( " ################ End Initializing Mod RMT UI Map Icons Script... ################ " );
end

Initialize();

